<?php
namespace Concrete\Package\CleanupDuplicateEntries\Job;

use Concrete\Core\Job\Job;
use Concrete\Core\Express\EntryList;
use Concrete\Core\Entity\Express\Entry;
use Doctrine\ORM\EntityManagerInterface;

class CleanupDuplicateEntries extends Job
{
    // ジョブの名前
    public function getJobName()
    {
        return t("Cleanup Duplicate illust_no Entries by Latest Modified Date");
    }

    // ジョブの説明
    public function getJobDescription()
    {
        return t("Deletes duplicate illust_no entries, keeping only the one with the most recent modification date.");
    }

    // ジョブの実行ロジック
    public function run()
    {
        // Express エンティティと属性のハンドル名
        $entityHandle = 'illust';
        $attributeHandle = 'illust_no';

        // Express エンティティを取得
        $express = \Core::make('express');
        $entity = $express->getObjectByHandle($entityHandle);

        if (!$entity) {
            // エンティティが見つからない場合はエラーメッセージを返す
            return t("Error: Express entity '{$entityHandle}' not found.");
        }

        // エンティティ内の全エントリを取得（パーミッション無視）
        $entryList = new EntryList($entity);
        $entryList->ignorePermissions();
        $entries = $entryList->getResults();

        // illust_no ごとにグループ化する配列を初期化
        $map = [];
        foreach ($entries as $entry) {
            $illustNo = $entry->getAttribute($attributeHandle);
            if (!$illustNo) {
                continue; // illust_no が空の場合はスキップ
            }

            if (!isset($map[$illustNo])) {
                $map[$illustNo] = [];
            }
            $map[$illustNo][] = $entry;
        }

        // Doctrine のエンティティマネージャを取得
        $em = \Core::make(EntityManagerInterface::class);
        $deleted = 0;

        // 重複グループを走査し、最新の1件を残して他を削除
        foreach ($map as $illustNo => $entryGroup) {
            if (count($entryGroup) > 1) {
                // getDateModified() を使って更新日時の降順にソート
                usort($entryGroup, function ($a, $b) {
                    return $b->getDateModified() <=> $a->getDateModified();
                });

                // 最新の1件を除く残りを削除対象に
                $entriesToDelete = array_slice($entryGroup, 1);

                foreach ($entriesToDelete as $entry) {
                    $found = $em->find(Entry::class, $entry->getID());
                    if ($found) {
                        $em->remove($found);
                        $deleted++;
                    }
                }
            }
        }

        // エンティティ削除を確定（コミット）
        $em->flush();

        // 削除件数を返す
        return "{$deleted} duplicate entries deleted.";
    }
}
